<?php
/**
 * Studiare Quiz Dashboard and Settings
 * Handles the dashboard and settings page for the Studiare Quiz plugin.
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

// Add menu item for dashboard page
function studiare_quiz_add_settings_menu() {
    // Check if Studiare Panel menu exists
    $parent_slug = menu_page_url('studiare_panel', false) ? 'studiare_panel' : null;

    if ($parent_slug) {
        // Add as submenu under Studiare Panel
        add_submenu_page(
            $parent_slug,
            __('Quiz Dashboard', 'studiare-core'),
            __('Quiz Dashboard', 'studiare-core'),
            'manage_options',
            'studiare_quiz_settings',
            'studiare_quiz_settings_page',
            8
        );
    } else {
        // Add as main menu
        add_menu_page(
            __('Quiz Builder', 'studiare-core'),
            __('Quiz Builder', 'studiare-core'),
            'manage_options',
            'sc-quiz',
            'studiare_quiz_settings_page',
            'dashicons-welcome-learn-more',
            30
        );
    }
}
add_action('admin_menu', 'studiare_quiz_add_settings_menu');

// Helper function to get quiz statistics
function studiare_quiz_get_stats() {
    // Check if stats are cached
    $cached_stats = get_transient('studiare_quiz_stats');
    if ($cached_stats !== false) {
        return $cached_stats;
    }

    global $wpdb;
    $table_name = $wpdb->prefix . 'studiare_quiz_results';

    // 1. Total number of quizzes
    $quiz_count = wp_count_posts('sc-quiz');
    $total_quizzes = isset($quiz_count->publish) ? $quiz_count->publish : 0;

    // 2. Total number of unique participants
    $participants = $wpdb->get_var(
        $wpdb->prepare(
            "SELECT COUNT(DISTINCT user_id) FROM %i WHERE user_id != %d",
            $table_name,
            0
        )
    );
    $total_participants = $participants ? $participants : 0;

    // 3. Pass percentage (optimized query)
    $results_stats = $wpdb->get_row(
        $wpdb->prepare(
            "SELECT 
                COUNT(*) as total_results,
                SUM(CASE 
                    WHEN score >= COALESCE(
                        (SELECT meta_value FROM {$wpdb->postmeta} WHERE post_id = quiz_id AND meta_key = '_quiz_passing_score'), 
                        70
                    ) THEN 1 
                    ELSE 0 
                END) as passed_results
            FROM %i",
            $table_name
        )
    );

    if ($results_stats && $results_stats->total_results > 0) {
        $pass_percentage = ($results_stats->passed_results / $results_stats->total_results) * 100;
        $pass_percentage = round($pass_percentage, 1);
    } else {
        $pass_percentage = 0;
    }

    $stats = [
        'total_quizzes' => $total_quizzes,
        'total_participants' => $total_participants,
        'pass_percentage' => $pass_percentage,
    ];

    // Cache the stats for 1 hour
    set_transient('studiare_quiz_stats', $stats, HOUR_IN_SECONDS);

    return $stats;
}

// Register settings
function studiare_quiz_register_settings() {
    // Existing settings
    register_setting('studiare_quiz_settings_group', 'studiare_quiz_post_type_enabled', [
        'type' => 'boolean',
        'default' => true,
        'sanitize_callback' => 'sanitize_text_field',
    ]);
    register_setting('studiare_quiz_settings_group', 'studiare_quiz_woocommerce_results_enabled', [
        'type' => 'boolean',
        'default' => false,
        'sanitize_callback' => 'sanitize_text_field',
    ]);
    register_setting('studiare_quiz_settings_group', 'studiare_quiz_admin_results_enabled', [
        'type' => 'boolean',
        'default' => true,
        'sanitize_callback' => 'sanitize_text_field',
    ]);

    // Settings for labels, slugs, has_archive, and public
    register_setting('studiare_quiz_settings_group', 'studiare_quiz_name', [
        'type' => 'string',
        'default' => __('Quizzes', 'studiare-core'),
        'sanitize_callback' => 'sanitize_text_field',
    ]);
    register_setting('studiare_quiz_settings_group', 'studiare_quiz_singular_name', [
        'type' => 'string',
        'default' => __('Quiz', 'studiare-core'),
        'sanitize_callback' => 'sanitize_text_field',
    ]);
    register_setting('studiare_quiz_settings_group', 'studiare_quiz_category_name', [
        'type' => 'string',
        'default' => __('Quiz Categories', 'studiare-core'),
        'sanitize_callback' => 'sanitize_text_field',
    ]);
    register_setting('studiare_quiz_settings_group', 'studiare_quiz_category_singular_name', [
        'type' => 'string',
        'default' => __('Quiz Category', 'studiare-core'),
        'sanitize_callback' => 'sanitize_text_field',
    ]);
    register_setting('studiare_quiz_settings_group', 'studiare_quiz_slug', [
        'type' => 'string',
        'default' => 'quizzes',
        'sanitize_callback' => 'sanitize_title',
    ]);
    register_setting('studiare_quiz_settings_group', 'studiare_quiz_category_slug', [
        'type' => 'string',
        'default' => 'quiz-category',
        'sanitize_callback' => 'sanitize_title',
    ]);
    register_setting('studiare_quiz_settings_group', 'studiare_quiz_has_archive', [
        'type' => 'boolean',
        'default' => true,
        'sanitize_callback' => 'sanitize_text_field',
    ]);
    register_setting('studiare_quiz_settings_group', 'studiare_quiz_public', [
        'type' => 'boolean',
        'default' => true,
        'sanitize_callback' => 'sanitize_text_field',
    ]);
}
add_action('admin_init', 'studiare_quiz_register_settings');

// Flush rewrite rules when options are updated
function studiare_quiz_flush_rewrite_rules($old_value, $new_value, $option = null) {
    if ($old_value !== $new_value) {
        if (is_null($option) || in_array($option, ['studiare_quiz_slug', 'studiare_quiz_category_slug', 'studiare_quiz_public'])) {
            flush_rewrite_rules();
        }
    }
}
add_action('update_option_studiare_quiz_woocommerce_results_enabled', 'studiare_quiz_flush_rewrite_rules', 10, 2);
add_action('update_option_studiare_quiz_post_type_enabled', 'studiare_quiz_flush_rewrite_rules', 10, 2);
add_action('update_option_studiare_quiz_admin_results_enabled', 'studiare_quiz_flush_rewrite_rules', 10, 2);
add_action('update_option_studiare_quiz_slug', 'studiare_quiz_flush_rewrite_rules', 10, 3);
add_action('update_option_studiare_quiz_category_slug', 'studiare_quiz_flush_rewrite_rules', 10, 3);
add_action('update_option_studiare_quiz_public', 'studiare_quiz_flush_rewrite_rules', 10, 3);

// Add WooCommerce account menu item for quiz results
function studiare_quiz_add_woocommerce_menu_item($items) {
    if (get_option('studiare_quiz_post_type_enabled', true) && get_option('studiare_quiz_woocommerce_results_enabled', false)) {
        $items['quiz-results'] = __('Quiz Results', 'studiare-core');
    }
    return $items;
}
add_filter('woocommerce_account_menu_items', 'studiare_quiz_add_woocommerce_menu_item');

// Conditionally add rewrite endpoint based on settings
function studiare_quiz_add_rewrite_endpoint() {
    if (get_option('studiare_quiz_post_type_enabled', true) && get_option('studiare_quiz_woocommerce_results_enabled', false)) {
        add_rewrite_endpoint('quiz-results', EP_ROOT | EP_PAGES);
    }
}
add_action('init', 'studiare_quiz_add_rewrite_endpoint');

// Add custom CSS for the WooCommerce My Account navigation icon directly to <head>
function studiare_quiz_add_my_account_styles() {
    if (class_exists('WooCommerce') && function_exists('is_account_page') && is_account_page() && get_option('studiare_quiz_post_type_enabled', true) && get_option('studiare_quiz_woocommerce_results_enabled', false)) {
        echo '<style type="text/css">
            .woocommerce-account .woocommerce-MyAccount-navigation ul li.woocommerce-MyAccount-navigation-link--quiz-results:before, i.quiz-results:before {
                content: "\\f681"; /* Font Awesome 5 Pro icon: fa-poll */
            }
        </style>';
    }
}
add_action('wp_head', 'studiare_quiz_add_my_account_styles');

// Display quiz results in WooCommerce account using shortcode
function studiare_quiz_woocommerce_results_content() {
    if (!get_option('studiare_quiz_post_type_enabled', true) || !get_option('studiare_quiz_woocommerce_results_enabled', false)) {
        wp_redirect(wc_get_page_permalink('myaccount'));
        exit;
    }
    echo do_shortcode('[studiare_quiz_results]');
}
add_action('woocommerce_account_quiz-results_endpoint', 'studiare_quiz_woocommerce_results_content');

// Dashboard and Settings page callback
function studiare_quiz_settings_page() {
    // Get current tab
    $current_tab = isset($_GET['tab']) ? sanitize_text_field($_GET['tab']) : 'dashboard';

    // Check if quiz post type is enabled
    $is_quiz_enabled = get_option('studiare_quiz_post_type_enabled', true);
    ?>
    <div class="wrap">
        <!-- Header -->
        <div class="studiare-quiz-header" style="background: linear-gradient(90deg, #1E3A8A, #3B82F6); padding: 20px; color: white; text-align: center; border-radius: 5px; margin-bottom: 20px;">
            <h1 style="text-shadow: 2px 2px 4px rgba(0,0,0,0.3); font-size: 2.5em; margin: 0; font-variation-settings: 'wght' 900; color: #fff;"><?php esc_html_e('Studiare Quiz Builder', 'studiare-core'); ?></h1>
            <p style="font-size: 1.2em; margin: 5px 0 0;"><?php esc_html_e('Developed by SunCode Team for Studiare Theme', 'studiare-core'); ?></p>
        </div>

        <!-- Tabs -->
        <h2 class="nav-tab-wrapper">
            <a href="?page=studiare_quiz_settings&tab=dashboard" class="nav-tab <?php echo $current_tab === 'dashboard' ? 'nav-tab-active' : ''; ?>"><?php esc_html_e('Dashboard', 'studiare-core'); ?></a>
            <a href="?page=studiare_quiz_settings&tab=settings" class="nav-tab <?php echo $current_tab === 'settings' ? 'nav-tab-active' : ''; ?>"><?php esc_html_e('Settings', 'studiare-core'); ?></a>
        </h2>

        <!-- Tab Content -->
        <?php if ($current_tab === 'dashboard') : ?>
            <?php if ($is_quiz_enabled) : ?>
                <?php $stats = studiare_quiz_get_stats(); ?>
                <!-- Statistics Widget -->
                <div class="studiare-quiz-stats" style="display: flex; gap: 20px; margin-top: 20px; margin-bottom: 40px; flex-wrap: wrap;">
                    <div style="flex: 1; min-width: 250px; background: linear-gradient(135deg, #ffffff, #e6f0ff); padding: 25px; border-radius: 10px; text-align: center; box-shadow: 0 4px 12px rgba(0,0,0,0.1); transition: transform 0.2s ease-in-out;">
                        <span class="dashicons dashicons-list-view" style="font-size: 40px; color: #1E3A8A; margin-bottom: 10px; display: block;"></span>
                        <h3 style="margin: 0 0 10px; color: #1E3A8A; font-size: 1.2em; font-weight: 600;"><?php esc_html_e('Total Quizzes', 'studiare-core'); ?></h3>
                        <p style="font-size: 2.5em; margin: 0 0 15px; color: #3B82F6; font-weight: bold;"><?php echo esc_html($stats['total_quizzes']); ?></p>
                    </div>
                    <div style="flex: 1; min-width: 250px; background: linear-gradient(135deg, #ffffff, #e6f0ff); padding: 25px; border-radius: 10px; text-align: center; box-shadow: 0 4px 12px rgba(0,0,0,0.1); transition: transform 0.2s ease-in-out;">
                        <span class="dashicons dashicons-groups" style="font-size: 40px; color: #1E3A8A; margin-bottom: 10px; display: block;"></span>
                        <h3 style="margin: 0 0 10px; color: #1E3A8A; font-size: 1.2em; font-weight: 600;"><?php esc_html_e('Participants', 'studiare-core'); ?></h3>
                        <p style="font-size: 2.5em; margin: 0; color: #3B82F6; font-weight: bold;"><?php echo esc_html($stats['total_participants']); ?></p>
                    </div>
                    <div style="flex: 1; min-width: 250px; background: linear-gradient(135deg, #ffffff, #e6f0ff); padding: 25px; border-radius: 10px; text-align: center; box-shadow: 0 4px 12px rgba(0,0,0,0.1); transition: transform 0.2s ease-in-out;">
                        <span class="dashicons dashicons-chart-pie" style="font-size: 40px; color: #1E3A8A; margin-bottom: 10px; display: block;"></span>
                        <h3 style="margin: 0 0 10px; color: #1E3A8A; font-size: 1.2em; font-weight: 600;"><?php esc_html_e('Pass Rate', 'studiare-core'); ?></h3>
                        <p style="font-size: 2.5em; margin: 0; color: #3B82F6; font-weight: bold;"><?php echo esc_html($stats['pass_percentage']); ?>%</p>
                    </div>
                </div>
                <!-- Navigation Links -->
                <div class="studiare-quiz-nav-links" style="display: flex; flex-wrap: wrap; gap: 10px; justify-content: center;">
                    <a href="<?php echo admin_url('edit.php?post_type=sc-quiz'); ?>" class="button" style="background: #3B82F6; border-color: #3B82F6; color: white; padding: 5px 10px; font-size: 0.9em;"><?php esc_html_e('View All Quizzes', 'studiare-core'); ?></a>
                    <a href="<?php echo admin_url('edit-tags.php?taxonomy=quiz_category&post_type=sc-quiz'); ?>" class="button" style="background: #3B82F6; border-color: #3B82F6; color: white; padding: 5px 10px; font-size: 0.9em;"><?php esc_html_e('Quiz Categories', 'studiare-core'); ?></a>
                    <?php if (get_option('studiare_quiz_admin_results_enabled', true)) : ?>
                        <a href="<?php echo admin_url('admin.php?page=studiare-quiz-results'); ?>" class="button" style="background: #3B82F6; border-color: #3B82F6; color: white; padding: 5px 10px; font-size: 0.9em;"><?php esc_html_e('View Quiz Results', 'studiare-core'); ?></a>
                    <?php endif; ?>
                    <a href="<?php echo admin_url('post-new.php?post_type=sc-quiz'); ?>" class="button" style="background: #4CAF50; border-color: #4CAF50; color: white; padding: 5px 10px; font-size: 0.9em;"><?php esc_html_e('Create New Quiz', 'studiare-core'); ?></a>
                </div>
            <?php else : ?>
                <div style="padding: 15px; margin-top: 20px; background: #ffd76a; border: 2px solid #ffba00; border-radius: 5px;">
                    <p><?php esc_html_e('The quiz system is currently disabled. Enable it in the Settings tab to manage quizzes.', 'studiare-core'); ?></p>
                </div>
            <?php endif; ?>
        <?php elseif ($current_tab === 'settings') : ?>
            <form method="post" action="options.php">
                <?php
                settings_fields('studiare_quiz_settings_group');
                do_settings_sections('studiare_quiz_settings_group');
                ?>
                <table class="form-table">
                    <tr>
                        <th scope="row"><?php esc_html_e('Enable Quiz System', 'studiare-core'); ?></th>
                        <td>
                            <label>
                                <input type="checkbox" name="studiare_quiz_post_type_enabled" value="1" <?php checked(get_option('studiare_quiz_post_type_enabled', true), true); ?> />
                                <?php esc_html_e('Enable or disable the quiz post type and related functionality.', 'studiare-core'); ?>
                            </label>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row"><?php esc_html_e('Enable Quiz Results in Admin', 'studiare-core'); ?></th>
                        <td>
                            <label>
                                <input type="checkbox" name="studiare_quiz_admin_results_enabled" value="1" <?php checked(get_option('studiare_quiz_admin_results_enabled', true), true); ?> />
                                <?php esc_html_e('Enable or disable the quiz results page in the admin dashboard.', 'studiare-core'); ?>
                            </label>
                        </td>
                    </tr>
                    <?php if (function_exists('WC')) : ?>
                        <tr>
                            <th scope="row"><?php esc_html_e('Show Quiz Results in WooCommerce Account', 'studiare-core'); ?></th>
                            <td>
                                <label>
                                    <input type="checkbox" name="studiare_quiz_woocommerce_results_enabled" value="1" <?php checked(get_option('studiare_quiz_woocommerce_results_enabled', false), true); ?> />
                                    <?php esc_html_e('Add a tab in the WooCommerce My Account page to display quiz results.', 'studiare-core'); ?>
                                </label>
                            </td>
                        </tr>
                    <?php endif; ?>
                    <tr>
                        <th scope="row"><label for="studiare_quiz_name"><?php esc_html_e('Quiz Plural Name', 'studiare-core'); ?></label></th>
                        <td>
                            <input type="text" id="studiare_quiz_name" name="studiare_quiz_name" value="<?php echo esc_attr(get_option('studiare_quiz_name', __('Quizzes', 'studiare-core'))); ?>" class="regular-text" />
                            <p class="description"><?php esc_html_e('Enter the plural name for quizzes (e.g., Quizzes).', 'studiare-core'); ?></p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row"><label for="studiare_quiz_singular_name"><?php esc_html_e('Quiz Singular Name', 'studiare-core'); ?></label></th>
                        <td>
                            <input type="text" id="studiare_quiz_singular_name" name="studiare_quiz_singular_name" value="<?php echo esc_attr(get_option('studiare_quiz_singular_name', __('Quiz', 'studiare-core'))); ?>" class="regular-text" />
                            <p class="description"><?php esc_html_e('Enter the singular name for a quiz (e.g., Quiz).', 'studiare-core'); ?></p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row"><label for="studiare_quiz_category_name"><?php esc_html_e('Quiz Category Plural Name', 'studiare-core'); ?></label></th>
                        <td>
                            <input type="text" id="studiare_quiz_category_name" name="studiare_quiz_category_name" value="<?php echo esc_attr(get_option('studiare_quiz_category_name', __('Quiz Categories', 'studiare-core'))); ?>" class="regular-text" />
                            <p class="description"><?php esc_html_e('Enter the plural name for quiz categories (e.g., Quiz Categories).', 'studiare-core'); ?></p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row"><label for="studiare_quiz_category_singular_name"><?php esc_html_e('Quiz Category Singular Name', 'studiare-core'); ?></label></th>
                        <td>
                            <input type="text" id="studiare_quiz_category_singular_name" name="studiare_quiz_category_singular_name" value="<?php echo esc_attr(get_option('studiare_quiz_category_singular_name', __('Quiz Category', 'studiare-core'))); ?>" class="regular-text" />
                            <p class="description"><?php esc_html_e('Enter the singular name for a quiz category (e.g., Quiz Category).', 'studiare-core'); ?></p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row"><label for="studiare_quiz_slug"><?php esc_html_e('Quiz Post Type Slug', 'studiare-core'); ?></label></th>
                        <td>
                            <input type="text" id="studiare_quiz_slug" name="studiare_quiz_slug" value="<?php echo esc_attr(get_option('studiare_quiz_slug', 'quizzes')); ?>" class="regular-text" />
                            <p class="description"><?php esc_html_e('Enter the URL slug for quizzes (e.g., quizzes). Changing this requires flushing permalinks.', 'studiare-core'); ?></p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row"><label for="studiare_quiz_category_slug"><?php esc_html_e('Quiz Category Slug', 'studiare-core'); ?></label></th>
                        <td>
                            <input type="text" id="studiare_quiz_category_slug" name="studiare_quiz_category_slug" value="<?php echo esc_attr(get_option('studiare_quiz_category_slug', 'quiz-category')); ?>" class="regular-text" />
                            <p class="description"><?php esc_html_e('Enter the URL slug for quiz categories (e.g., quiz-category). Changing this requires flushing permalinks.', 'studiare-core'); ?></p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row"><?php esc_html_e('Enable Quiz Archive', 'studiare-core'); ?></th>
                        <td>
                            <label>
                                <input type="checkbox" name="studiare_quiz_has_archive" value="1" <?php checked(get_option('studiare_quiz_has_archive', true), true); ?> />
                                <?php esc_html_e('Enable or disable the archive page for quizzes.', 'studiare-core'); ?>
                            </label>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row"><?php esc_html_e('Enable Public Access', 'studiare-core'); ?></th>
                        <td>
                            <label>
                                <input type="checkbox" name="studiare_quiz_public" value="1" <?php checked(get_option('studiare_quiz_public', true), true); ?> />
                                <?php esc_html_e('Enable or disable public access to single quiz and category pages. If disabled, quizzes are only accessible via shortcodes or widgets.', 'studiare-core'); ?>
                            </label>
                        </td>
                    </tr>
                </table>
                <?php submit_button(__('Save Settings', 'studiare-core')); ?>
            </form>
        <?php endif; ?>

        <!-- Inline CSS for hover effect and form styling -->
        <style>
            .studiare-quiz-stats div:not(.studiare-quiz-nav-links):hover {
                transform: translateY(-5px);
                box-shadow: 0 6px 16px rgba(0,0,0,0.15);
            }
            .form-table th, .form-table td {
                padding: 15px 10px;
            }
            .form-table input[type="text"] {
                width: 300px;
            }
        </style>
    </div>
    <?php
}
?>