<?php
/**
 * Studiare Quiz Meta Boxes
 * Defines meta boxes for the sc-quiz post type
 */

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly
}

/**
 * Add meta boxes for Quiz post type
 */
function studiare_add_quiz_meta_boxes() {
    // Add settings meta box
    add_meta_box(
        'quiz_settings_meta_box',
        __('Quiz Settings', 'studiare-core'),
        'studiare_render_quiz_settings_meta_box',
        'sc-quiz',
        'normal',
        'high'
    );
}
add_action('add_meta_boxes', 'studiare_add_quiz_meta_boxes');

/**
 * Render the Quiz Settings meta box
 */
function studiare_render_quiz_settings_meta_box($post) {
    // Nonce field for security
    wp_nonce_field('studiare_save_quiz_settings', 'studiare_quiz_settings_nonce');

    // Get stored values
    $widget_description = get_post_meta($post->ID, '_quiz_widget_description', true);
    $description = get_post_meta($post->ID, '_quiz_description', true);
    $start_time = get_post_meta($post->ID, '_quiz_start_time', true);
    $end_time = get_post_meta($post->ID, '_quiz_end_time', true);
    $answer_time = get_post_meta($post->ID, '_quiz_answer_time', true);
    $negative_marking = get_post_meta($post->ID, '_quiz_negative_marking', true);
    $enable_pagination = get_post_meta($post->ID, '_quiz_enable_pagination', true);
    $questions_per_page = get_post_meta($post->ID, '_quiz_questions_per_page', true);
    $restrict_to_members = get_post_meta($post->ID, '_quiz_restrict_to_members', true);
    $restrict_to_products = get_post_meta($post->ID, '_quiz_restrict_to_products', true);
    $restrict_to_products = is_array($restrict_to_products) ? $restrict_to_products : [];
    $max_attempts = get_post_meta($post->ID, '_quiz_max_attempts', true);
    $score_storage_method = get_post_meta($post->ID, '_quiz_score_storage_method', true);
    $passing_score = get_post_meta($post->ID, '_quiz_passing_score', true);

    // Convert stored time to datetime-local compatible format (if not empty)
    $start_time_formatted = $start_time ? date('Y-m-d\TH:i', strtotime($start_time)) : '';
    $end_time_formatted = $end_time ? date('Y-m-d\TH:i', strtotime($end_time)) : '';
    ?>
    <p>
        <label for="quiz_widget_description"><strong><?php _e('Description for Widget Display in Studiar Theme Lessons:', 'studiare-core'); ?></strong></label><br>
        <input type="text" id="quiz_widget_description" name="quiz_widget_description" value="<?php echo esc_attr($widget_description); ?>" style="width: 100%;" placeholder="<?php esc_attr_e('Enter a short description for the widget', 'studiare-core'); ?>">
    </p>

    <p>
        <label for="quiz_description"><strong><?php _e('Quiz Description:', 'studiare-core'); ?></strong></label><br>
        <?php
        wp_editor(
            wp_kses_post($description),
            'quiz_description',
            [
                'textarea_name' => 'quiz_description',
                'media_buttons' => true,
                'teeny' => false,
                'textarea_rows' => 5,
            ]
        );
        ?>
    </p>

    <p>
        <label for="quiz_start_time"><strong><?php _e('Start Time:', 'studiare-core'); ?></strong></label><br>
        <input type="datetime-local" id="quiz_start_time" name="quiz_start_time" value="<?php echo esc_attr($start_time_formatted); ?>" style="width: 100%;">
    </p>

    <p>
        <label for="quiz_end_time"><strong><?php _e('End Time:', 'studiare-core'); ?></strong></label><br>
        <input type="datetime-local" id="quiz_end_time" name="quiz_end_time" value="<?php echo esc_attr($end_time_formatted); ?>" style="width: 100%;">
    </p>

    <p>
        <label for="quiz_answer_time"><strong><?php _e('Answer Duration (minutes):', 'studiare-core'); ?></strong></label><br>
        <input type="number" id="quiz_answer_time" name="quiz_answer_time" value="<?php echo esc_attr($answer_time); ?>" min="1" style="width: 100%;">
    </p>

    <p>
        <label for="quiz_negative_marking">
            <input type="checkbox" id="quiz_negative_marking" name="quiz_negative_marking" value="yes" <?php checked($negative_marking, 'yes'); ?>>
            <?php _e('Enable Negative Marking', 'studiare-core'); ?>
        </label>
    </p>

    <p>
        <label for="quiz_enable_pagination">
            <input type="checkbox" id="quiz_enable_pagination" name="quiz_enable_pagination" value="yes" <?php checked($enable_pagination, 'yes'); ?>>
            <?php _e('Enable Pagination', 'studiare-core'); ?>
        </label>
    </p>

    <p>
        <label for="quiz_questions_per_page"><strong><?php _e('Questions per Page:', 'studiare-core'); ?></strong></label><br>
        <input type="number" id="quiz_questions_per_page" name="quiz_questions_per_page" value="<?php echo esc_attr($questions_per_page ? $questions_per_page : 5); ?>" min="1" style="width: 100%;">
    </p>

    <p>
        <label for="quiz_restrict_to_members">
            <input type="checkbox" id="quiz_restrict_to_members" name="quiz_restrict_to_members" value="yes" <?php checked($restrict_to_members, 'yes'); ?>>
            <?php _e('Restrict to Logged-in Members Only', 'studiare-core'); ?>
        </label>
    </p>

    <p>
        <label for="quiz_restrict_to_products"><strong><?php _e('Restrict to Purchasers of Products:', 'studiare-core'); ?></strong></label><br>
        <select id="quiz_restrict_to_products" name="quiz_restrict_to_products[]" multiple="multiple" style="width: 100%;">
            <?php
            $products = wc_get_products([
                'status' => 'publish',
                'limit' => -1,
                'orderby' => 'title',
                'order' => 'ASC',
            ]);
            foreach ($products as $product) {
                printf(
                    '<option value="%s" %s>%s</option>',
                    esc_attr($product->get_id()),
                    in_array($product->get_id(), $restrict_to_products) ? 'selected' : '',
                    esc_html($product->get_name())
                );
            }
            ?>
        </select>
    </p>

    <p>
        <label for="quiz_max_attempts"><strong><?php _e('Maximum Attempts (0 for unlimited):', 'studiare-core'); ?></strong></label><br>
        <input type="number" id="quiz_max_attempts" name="quiz_max_attempts" value="<?php echo esc_attr($max_attempts ? $max_attempts : 0); ?>" min="0" style="width: 100%;">
    </p>

    <p>
        <label for="quiz_score_storage_method"><strong><?php _e('Score Storage Method:', 'studiare-core'); ?></strong></label><br>
        <select id="quiz_score_storage_method" name="quiz_score_storage_method" style="width: 100%;">
            <option value="all_attempts" <?php selected($score_storage_method, 'all_attempts'); ?>><?php _e('Store All Attempts', 'studiare-core'); ?></option>
            <option value="highest_score" <?php selected($score_storage_method, 'highest_score'); ?>><?php _e('Store Highest Score', 'studiare-core'); ?></option>
            <option value="average_score" <?php selected($score_storage_method, 'average_score'); ?>><?php _e('Store Average Score', 'studiare-core'); ?></option>
        </select>
    </p>

    <p>
        <label for="quiz_passing_score"><strong><?php _e('Passing Score:', 'studiare-core'); ?></strong></label><br>
        <input type="number" id="quiz_passing_score" name="quiz_passing_score" value="<?php echo esc_attr($passing_score ? $passing_score : 0); ?>" min="0" step="0.1" style="width: 100%;">
        <small><?php _e('Enter the minimum score required to pass the quiz (e.g., 70 out of 100).', 'studiare-core'); ?></small>
    </p>

    <script>
        jQuery(document).ready(function($) {
            $('#quiz_restrict_to_products').select2({
                placeholder: '<?php esc_js(__('Select products', 'studiare-core')); ?>',
                allowClear: true
            });
        });
    </script>
    <?php
}

/**
 * Save meta box values
 */
function studiare_save_quiz_meta_box_data($post_id) {
    // Check nonce
    if (!isset($_POST['studiare_quiz_settings_nonce']) || !wp_verify_nonce($_POST['studiare_quiz_settings_nonce'], 'studiare_save_quiz_settings')) {
        return;
    }

    // Avoid autosave
    if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) {
        return;
    }

    // Check user permissions
    if (!current_user_can('edit_post', $post_id)) {
        return;
    }

    // Sanitize and save widget description
    if (isset($_POST['quiz_widget_description'])) {
        update_post_meta($post_id, '_quiz_widget_description', sanitize_text_field($_POST['quiz_widget_description']));
    } else {
        delete_post_meta($post_id, '_quiz_widget_description');
    }

    // Sanitize and save other settings
    if (isset($_POST['quiz_description'])) {
        update_post_meta($post_id, '_quiz_description', wp_kses_post($_POST['quiz_description']));
    } else {
        delete_post_meta($post_id, '_quiz_description');
    }

    if (isset($_POST['quiz_start_time'])) {
        // Convert datetime-local to 'Y-m-d H:i:s' for storage
        $start_time = sanitize_text_field($_POST['quiz_start_time']);
        $start_time = $start_time ? date('Y-m-d H:i:s', strtotime($start_time)) : '';
        update_post_meta($post_id, '_quiz_start_time', $start_time);
    } else {
        delete_post_meta($post_id, '_quiz_start_time');
    }

    if (isset($_POST['quiz_end_time'])) {
        // Convert datetime-local to 'Y-m-d H:i:s' for storage
        $end_time = sanitize_text_field($_POST['quiz_end_time']);
        $end_time = $end_time ? date('Y-m-d H:i:s', strtotime($end_time)) : '';
        update_post_meta($post_id, '_quiz_end_time', $end_time);
    } else {
        delete_post_meta($post_id, '_quiz_end_time');
    }

    if (isset($_POST['quiz_answer_time'])) {
        update_post_meta($post_id, '_quiz_answer_time', intval($_POST['quiz_answer_time']));
    } else {
        delete_post_meta($post_id, '_quiz_answer_time');
    }

    update_post_meta($post_id, '_quiz_negative_marking', isset($_POST['quiz_negative_marking']) ? 'yes' : 'no');
    update_post_meta($post_id, '_quiz_enable_pagination', isset($_POST['quiz_enable_pagination']) ? 'yes' : 'no');

    if (isset($_POST['quiz_questions_per_page'])) {
        update_post_meta($post_id, '_quiz_questions_per_page', intval($_POST['quiz_questions_per_page']));
    } else {
        delete_post_meta($post_id, '_quiz_questions_per_page');
    }

    update_post_meta($post_id, '_quiz_restrict_to_members', isset($_POST['quiz_restrict_to_members']) ? 'yes' : 'no');

    if (isset($_POST['quiz_restrict_to_products'])) {
        $products = array_map('intval', (array)$_POST['quiz_restrict_to_products']);
        update_post_meta($post_id, '_quiz_restrict_to_products', $products);
    } else {
        delete_post_meta($post_id, '_quiz_restrict_to_products');
    }

    if (isset($_POST['quiz_max_attempts'])) {
        update_post_meta($post_id, '_quiz_max_attempts', max(0, intval($_POST['quiz_max_attempts']))); // Ensure non-negative
    } else {
        update_post_meta($post_id, '_quiz_max_attempts', 0); // Default to unlimited
    }

    if (isset($_POST['quiz_score_storage_method'])) {
        $valid_methods = ['all_attempts', 'highest_score', 'average_score'];
        $method = in_array($_POST['quiz_score_storage_method'], $valid_methods) ? $_POST['quiz_score_storage_method'] : 'all_attempts';
        update_post_meta($post_id, '_quiz_score_storage_method', $method);
    } else {
        update_post_meta($post_id, '_quiz_score_storage_method', 'all_attempts'); // Default
    }

    if (isset($_POST['quiz_passing_score'])) {
        update_post_meta($post_id, '_quiz_passing_score', floatval($_POST['quiz_passing_score']));
    } else {
        update_post_meta($post_id, '_quiz_passing_score', 0); // Default to 0
    }
}
add_action('save_post_sc-quiz', 'studiare_save_quiz_meta_box_data');

/**
 * Register CMB2 meta box for quiz questions
 */
function studiare_register_questions_meta_box() {
    // Check if CMB2 is loaded
    if (!class_exists('CMB2')) {
        // Add admin notice for debugging
        add_action('admin_notices', function() {
            global $pagenow, $post_type;
            // Only show notice on quiz edit screen
            if ($pagenow === 'post.php' && isset($_GET['post']) && get_post_type($_GET['post']) === 'sc-quiz' || 
                $pagenow === 'post-new.php' && isset($_GET['post_type']) && $_GET['post_type'] === 'sc-quiz') {
                ?>
                <div class="notice notice-error is-dismissible">
                    <p><?php _e('CMB2 is not available. Quiz questions meta box will not be displayed. Please ensure CMB2 is installed and loaded.', 'studiare-core'); ?></p>
                </div>
                <?php
            }
        });
        return;
    }

    $prefix = '_quiz_';

    $cmb = new_cmb2_box(array(
        'id'           => $prefix . 'questions_box',
        'title'        => __('Quiz Questions', 'studiare-core'),
        'object_types' => array('sc-quiz'),
        'context'      => 'normal',
        'priority'     => 'high',
        'show_names'   => true,
    ));

    $group_field_id = $cmb->add_field(array(
        'id'          => $prefix . 'questions',
        'type'        => 'group',
        'description' => __('Add questions for the quiz.', 'studiare-core'),
        'options'     => array(
            'group_title'   => __('Question {#}', 'studiare-core'),
            'add_button'    => __('Add New Question', 'studiare-core'),
            'remove_button' => __('Remove Question', 'studiare-core'),
            'sortable'      => true,
            'closed'        => false,
        ),
    ));

    // Fields inside the group
    $cmb->add_group_field($group_field_id, array(
        'name' => __('Question Title', 'studiare-core'),
        'id'   => 'title',
        'type' => 'text',
        'attributes' => array(
            'required' => 'required',
        ),
    ));

    $cmb->add_group_field($group_field_id, array(
        'name' => __('Question Description', 'studiare-core'),
        'id'   => 'description',
        'type' => 'textarea_small',
        'description' => __('Optional description for the question.', 'studiare-core'),
    ));

    $cmb->add_group_field($group_field_id, array(
        'name' => __('Question Image', 'studiare-core'),
        'id'   => 'image',
        'type' => 'file',
        'description' => __('Optional image for the question.', 'studiare-core'),
        'options' => array(
            'url' => false, // Hide the URL field
        ),
    ));

    $cmb->add_group_field($group_field_id, array(
        'name' => __('Option A', 'studiare-core'),
        'id'   => 'option_a',
        'type' => 'text',
        'attributes' => array(
            'required' => 'required',
        ),
    ));

    $cmb->add_group_field($group_field_id, array(
        'name' => __('Option B', 'studiare-core'),
        'id'   => 'option_b',
        'type' => 'text',
        'attributes' => array(
            'required' => 'required',
        ),
    ));

    $cmb->add_group_field($group_field_id, array(
        'name' => __('Option C', 'studiare-core'),
        'id'   => 'option_c',
        'type' => 'text',
        'attributes' => array(
            'required' => 'required',
        ),
    ));

    $cmb->add_group_field($group_field_id, array(
        'name' => __('Option D', 'studiare-core'),
        'id'   => 'option_d',
        'type' => 'text',
        'attributes' => array(
            'required' => 'required',
        ),
    ));

    $cmb->add_group_field($group_field_id, array(
        'name'    => __('Correct Answer', 'studiare-core'),
        'id'      => 'correct',
        'type'    => 'select',
        'options' => array(
            'a' => __('A', 'studiare-core'),
            'b' => __('B', 'studiare-core'),
            'c' => __('C', 'studiare-core'),
            'd' => __('D', 'studiare-core'),
        ),
        'attributes' => array(
            'required' => 'required',
        ),
    ));

    $cmb->add_group_field($group_field_id, array(
        'name' => __('Score', 'studiare-core'),
        'id'   => 'score',
        'type' => 'text_small',
        'attributes' => array(
            'type' => 'number',
            'step' => '0.1',
            'min'  => '0',
            'required' => 'required',
        ),
    ));
}
add_action('cmb2_admin_init', 'studiare_register_questions_meta_box', 10);
?>